package checkers.controller;
/**
 * Facade.java
 *
 * Version	
 *   $Id: GameController.java,v 1.17 2005/11/08 23:02:10 pjk7060 Exp $
 *
 * Revisions:
 *   $Log: GameController.java,v $
 *   Revision 1.17  2005/11/08 23:02:10  pjk7060
 *   comments
 *
 *   Revision 1.16  2005/11/08 22:37:00  pjk7060
 *   removed unused timer code
 *
 *   fixed warnings
 *
 *   Revision 1.15  2005/11/08 06:00:38  pjk7060
 *   removed timer stuff
 *
 *   Revision 1.14  2005/11/07 00:01:30  mtg9625
 *   Draw functionality implemented, at least locally
 *
 *   Revision 1.13  2005/11/06 23:16:06  mtg9625
 *   Added methods to deal with a draw
 *
 *   Revision 1.12  2005/11/06 22:58:54  pjk7060
 *   made clicking the x resign from the game
 *
 *   Revision 1.11  2005/11/06 22:14:24  mtg9625
 *   Checks if a player has won and displays message
 *   Resign button works and ends game
 *
 *   Revision 1.10  2005/11/06 21:52:15  pjk7060
 *   fixed network play colors not being assigned correctly
 *
 *   Revision 1.9  2005/11/06 20:18:54  pjk7060
 *   Fixed turn issues
 *
 *   Revision 1.8  2005/11/06 19:38:46  pjk7060
 *   worked many bugs out of networking
 *
 *   Revision 1.7  2005/11/05 17:42:09  pjk7060
 *   working on update, displays pieces and current turn
 *
 *   Revision 1.6  2005/11/05 15:35:32  pjk7060
 *   Little fixes
 *
 *   Revision 1.5  2005/11/04 20:04:13  mtg9625
 *   now displays completely, still need to do update method
 *   fixed outofbounds error in board, (added a - 1)
 *   fixed nullPointerException in CheckersGame(didn't initialize array and tried to access)
 *   removed imports of deleted classes in GameController
 *
 *   Revision 1.4  2005/11/04 15:50:16  pjk7060
 *   Fixed registering of components with proper observers, and proper references
 *
 *   Revision 1.3  2005/11/04 04:05:37  pjk7060
 *   Created Global Constants interface
 *
 *   Further development on controller, and initial game setup
 *
 *   Revision 1.2  2005/11/02 04:34:05  pjk7060
 *   moved classes into the proper packages
 *
 *   Revision 1.1  2005/11/02 04:31:21  pjk7060
 *   split everything into packages
 *
 *   implemented the networking class
 *
 *   Revision 1.2  2005/10/29 14:22:31  jmr1040
 *   Fixed formatting (so we can see all nesting)
 *
 *   Revision 1.1  2005/10/29 14:16:12  jmr1040
 *   Renamed Driver and Facade to match purpose
 *
 *   Revision 1.1  2005/10/20 13:53:35  jmr1040
 *   Initial state of Checkers project
 *
 *   Revision 1.1  2002/10/22 21:12:52  se362
 *   Initial creation of case study
 *
 */


import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;

import checkers.CheckersConstants;
import checkers.model.CheckersGame;
import checkers.model.NetPacket;

/**
 * An interface between the GUI and the kernel classes in a checkers game.
 * 
 * @author
 */

public class GameController implements ActionListener, CheckersConstants, WindowListener {
	
	// Model
	private CheckersGame game;
	
	/**
	 * Creates the controller with a reference to the game.
	 * 
	 * @param game CheckersGame
	 */
	public GameController(CheckersGame game) {
		this.game = game;
	}

	/**
	 * Handles events
	 */
	public void actionPerformed(ActionEvent e) {	
		if (e.getActionCommand().equals(ACCEPT)){
			//does sequence of events to accept a draw
			pressAcceptDraw();
		}else if (e.getActionCommand().equals(DRAW)) {
			// does sequence of events to offer a draw
			pressDraw();				
			
			// if resign is pressed
		} else if (e.getActionCommand().equals(RESIGN)) {
			// does sequence of events for a resign
			pressQuit();
			
			// if the event should be an integer of the square clicked
		} else if (e.getActionCommand().equals(PLAYER1SET)){
			
			setPlayerName(0, ((NetPacket)e.getSource()).getPlayer1());
			game.setPlayer1Color(((NetPacket)e.getSource()).getPlayer1Color());
			startGame();
			
		} else if (e.getActionCommand().equals(PLAYER2SET)){
			
			setPlayerName(1, ((NetPacket)e.getSource()).getPlayer2());			
			startGame();
			
		} else {
			
			try{
				int space = Integer.parseInt(e.getActionCommand());
				if (space <= 32 && space >= 0){
					selectSpace(space);
				}
			}catch(NumberFormatException e2){
				System.err.println("Invalid ActionEvent");
			}
			
		}		
	}
	
	/**
	 * This method should be called to select a space on the board, either as
	 * the starting point or the ending point for a move.  Forwards the 
	 * request to game.
	 *  
	 * @param space
	 *            an int indicating which space to move to, according to the
	 *            standard checkers numbering scheme, left to right and top to
	 *            bottom.
	 */
	public void selectSpace(int space) {
		game.selectSpace(space);
	}

	/**
	 * Tell the kernel that the user has quit/resigned the game or quit the
	 * program
	 */
	public void pressQuit() {

		// Alert players and the kernel that one person
		// has quit calls quitGame() for both players
		game.endInQuit(game.getCurrentTurn());

	}

	/**
	 * Tell the model that the user has requested a draw.
	 */
	public void pressDraw() {

		// Alerts both players and the kernel that one person
		// has offered a draw calls offerDraw() on both players
		game.offerDraw();

	}

	/**
	 * Tell the model that the user has accepted a draw.
	 * 
	 */
	public void pressAcceptDraw() {

		// calls acceptDraw() in the driver
		game.endInDraw(game.getCurrentTurn());
	}

	/**
	 * Tell the model to associate the given name with the given player number.
	 * 
	 * @param playerNum
	 *            the number of a player
	 * @param name
	 *            the name that player should be given
	 * 
	 * @pre playerNum is a valid player number
	 */
	public void setPlayerName(int playerNum, String name) {		
		game.setPlayerName(playerNum, name);
	}

	/**
	 * Display to local players that the game has ended with the message
	 * provided.
	 * 
	 * @param message
	 * 
	 * @post the game ends
	 */
	public void showEndGame(String message) {
		// make sure game is over
		// calls endGame in driver object
		game.endGame(message);
	}

	/**
	 * Called when both players have clicked OK on the end game dialog box
	 * 
	 * @post the game has ended
	 */
	public void endGameAccepted() {

		// waits until both players have accepted the end of the game
		// end the game
	}


	/**
	 * Begin the game
	 */
	public void startGame() {
		game.startGame();		
	}
	
	/**
	 * Sets the playerNum for a networked game. 
	 * (so that you can't move the other player's pieces)
	 * 
	 * @param playerNum number of player
	 */
	public void setPlayer(int playerNum){
		game.setPlayerNum(playerNum);
	}

	/**
	 * Called when the user tries to exit the program.
	 */
	public void windowClosing(WindowEvent arg0) {
		if (game.isGameActive()){
			pressQuit();
		}
		System.exit(0);
	}
	
	// Necessary methods when implementing WindowListener
	
	public void windowOpened(WindowEvent arg0) {
	}

	public void windowClosed(WindowEvent arg0) {
	}

	public void windowIconified(WindowEvent arg0) {
	}

	public void windowDeiconified(WindowEvent arg0) {
	}

	public void windowActivated(WindowEvent arg0) {
	}

	public void windowDeactivated(WindowEvent arg0) {
	}
}