package checkers.model;
import java.util.Observable;
import java.util.Vector;

import checkers.CheckersConstants;

/**
 * This class represents the board on which checkers is being played. The board
 * holds a collection of pieces.
 * 
 * @invariant all variables have valid values
 * 
 * @author
 */
public class Board extends Observable implements CheckersConstants {

	private static final int NUM_PIECES = 12; // number of pieces per side

	private Piece pieces[]; // the pieces that are on the board

	/**
	 * This constructor creates a new board at the beginning of the game
	 */

	public Board() {

		// create a array of size 32, generate piece objects and
		// put them in the correct location in the array
		pieces = new Piece[NUM_SPACES];

		// create blue and white pieces
		for (int i = 0; i < NUM_PIECES; i++) {
			pieces[i] = new Piece(BLUE, SINGLE);
			pieces[NUM_SPACES - i - 1] = new Piece(WHITE, SINGLE);
		}
	}

	/**
	 * Move the piece at the start position to the end position
	 * 
	 * @param theMove -
	 *            current location of the piece
	 * @return whether the piece was successfully moved
	 * 
	 * @pre Move has been validated
	 */
	public boolean movePiece(Move theMove) {

		boolean returnValue = false; // starts false, assuming invalid move

		// check if the end position of the piece is occupied
		if (!occupied(theMove.endLocation())) {
			// if it is not set a start position in the array to null
			// put a piece object at the end position in the array
			// that was at the start positon before
			pieces[theMove.endLocation()] = pieces[theMove.startLocation()];
			pieces[theMove.startLocation()] = null;
			// return true on successful move
			returnValue = true;
			setChanged();
			notifyObservers();
		}
		return returnValue;
	}

	/**
	 * This method checks if the space on the board contains a piece
	 * 
	 * @param space -
	 *            the space that needs to be checked
	 * 
	 * @return true or false depending on the situation
	 */
	public boolean occupied(int space) {

		boolean returnValue = false;

		// if it's in the bounds of the array,
		// return true if the space is occupied
		// false if the space is empty
		// if it's outside the bounds of the array,
		// return true

		if (space >= 0 && space < 32 && pieces[space] != null) {
			returnValue = true;
		}

		return returnValue;

	}

	/**
	 * This method removes piece at the position space
	 * 
	 * @param space -
	 *            the positon of the piece to be removed
	 * @pre the space is occupied
	 * @post the space is no longer occupied
	 */
	public void removePiece(int space) {
		
		// only if the space has a piece
		//if (occupied(space))
			// go to the space position in the array
			// set it equal to null
			pieces[space] = null;
	}

	/**
	 * This method creates a king piece
	 * 
	 * @param space -
	 *            the position at which the king piece is created
	 * @pre the space is occupied
	 * @post the piece is now a king
	 */
	public void kingPiece(int space) {

		if (occupied(space)) {
			// create a new king piece
			// go to the space position in the array and place it there
			// if the position is not ocupied
			//int color = pieces[space].getColor();
			//Piece piece = new KingPiece(color);
			pieces[space].setType(KING);
			setChanged();
			notifyObservers();
		}
	}

	/**
	 * This method returns the color of the piece at a certain space
	 * 
	 * @param space -
	 *            the position of the piece on the board
	 * 
	 * @return the color of this piece, -1 if empty
	 * @pre the space is occupied
	 */
	public int colorAt(int space) {

		int returnValue = -1;
		// go to the space position in the array
		// check if there is a piece at that position
		// if there is none, return null
		// else retrun the color of the piece

		if (occupied(space)) {
			returnValue = pieces[space].getColor();
		}

		return returnValue;

	}

	/**
	 * This method returns the piece at the certain position
	 * 
	 * @param space -
	 *            the space of the piece
	 * 
	 * @return the piece at that space, null if space is empty
	 * @pre the space is occupied
	 */
	public Piece getPieceAt(int space) {
		Piece returnValue = null;

		if (space >=0 && space < NUM_SPACES && occupied(space)) {
			// check if there is piece at space position
			// if there is none, return null
			// else return the piece at that position
			returnValue = pieces[space];
		}
		return returnValue;
	}

	/**
	 * This method returns if there is a piece of color on the board
	 * 
	 * @param color -
	 *            the color of the piece (as defined by constants in Piece)
	 * 
	 * @return true if there is a piece of color left on the board else return
	 *         false
	 */
	public boolean hasPieceOf(int color) {

		boolean returnValue = false;

		// go through the whole array
		// if there is a piece of color in the arrar return true
		// else return false
		for (int i = 1; i < pieces.length; i++) {
			if (occupied(i) && pieces[i].getColor() == color) {
				returnValue = true;
				i = pieces.length;
			}
		}

		return returnValue;

	}

	/**
	 * This method returns the size of the board
	 * 
	 * @return the size of the board, always same as size of piece array
	 */
	public int sizeOf() {
		return pieces.length;
	}

	/**
	 * This method is a replacement for bluePieces and whitePieces. It is more general
	 * and will return a vector of all pieces of the color passed as a parameter
	 * 
	 * @param color Color of pieces to get (as defined by constants in class Piece)
	 * 
	 * @return a vector of Pieces of the given color
	 */
	public Vector getPieces(int color) {
		Vector retval = new Vector();
		for( int i=0; i<pieces.length; ++i){
			if(occupied(i) && pieces[i].getColor() == color) {
				retval.add(pieces[i]);
			}
		}
		return retval;
	}
	
}// Board

