package checkers.model;

/*
 * CheckersGame.java
 *
 * Version
 *    $Id: CheckersGame.java,v 1.20 2005/11/09 02:21:24 pjk7060 Exp $ 
 *
 * Revisions:
 *    $Log: CheckersGame.java,v $
 *    Revision 1.20  2005/11/09 02:21:24  pjk7060
 *    comment change
 *
 *    Revision 1.19  2005/11/08 23:02:09  pjk7060
 *    comments
 *
 *    Revision 1.18  2005/11/08 22:37:00  pjk7060
 *    removed unused timer code
 *
 *    fixed warnings
 *
 *    Revision 1.17  2005/11/07 00:01:30  mtg9625
 *    Draw functionality implemented, at least locally
 *
 *    Revision 1.16  2005/11/06 22:46:08  pjk7060
 *    fixed resigning over the network
 *
 *    Revision 1.15  2005/11/06 22:14:24  mtg9625
 *    Checks if a player has won and displays message
 *    Resign button works and ends game
 *
 *    Revision 1.14  2005/11/06 21:52:15  pjk7060
 *    fixed network play colors not being assigned correctly
 *
 *    Revision 1.13  2005/11/06 20:18:54  pjk7060
 *    Fixed turn issues
 *
 *    Revision 1.12  2005/11/06 19:38:46  pjk7060
 *    worked many bugs out of networking
 *
 *    Revision 1.11  2005/11/05 19:09:53  jim5870
 *    fixed
 *
 *    Revision 1.10  2005/11/05 18:42:19  pjk7060
 *    fixed jumps
 *
 *    Revision 1.9  2005/11/05 18:24:32  pjk7060
 *    forced jump fix
 *
 *    Revision 1.8  2005/11/05 17:42:09  pjk7060
 *    working on update, displays pieces and current turn
 *
 *    Revision 1.7  2005/11/05 15:57:26  jmr1040
 *    *** empty log message ***
 *
 *    Revision 1.6  2005/11/05 15:55:32  jmr1040
 *    *** empty log message ***
 *
 *    Revision 1.5  2005/11/04 20:04:34  mtg9625
 *    now displays completely, still need to do update method
 *    fixed outofbounds error in board, (added a - 1)
 *    fixed nullPointerException in CheckersGame(didn't initialize array and tried to access)
 *    removed imports of deleted classes in GameController
 *
 *    Revision 1.4  2005/11/04 16:03:25  jmr1040
 *    Everything's working?
 *
 *    Revision 1.3  2005/11/04 00:45:47  jmr1040
 *    *** empty log message ***
 *
 *    Revision 1.2  2005/11/03 00:24:14  jmr1040
 *    Modified to work with colors as Ints
 *    Commented out endTurn until ready to work with it
 *    Modified class to track players to be able to remove Player class
 *
 *    Revision 1.2  2005/10/29 14:22:31  jmr1040
 *    Fixed formatting (so we can see all nesting)
 *
 *    Revision 1.1  2005/10/29 14:16:12  jmr1040
 *    Renamed Driver and Facade to match purpose
 *
 *    Revision 1.1  2005/10/20 13:53:36  jmr1040
 *    Initial state of Checkers project
 *
 *    Revision 1.1  2002/10/22 21:12:52  se362
 *    Initial creation of case study
 *
 */

import java.util.Observable;
import checkers.CheckersConstants;

/**
 * 
 * This class is a part of the main functionality of the checkers game. Its 
 * functions include knowing whose turn it is, remembering multiple jumps, 
 * relaying end of game conditions and ending the game.
 * 
 * @author
 * 
 */

public class CheckersGame extends Observable implements CheckersConstants {
	
	// how many players
	private int numPlayers;

	// their names
	private String[] playerNames;

	// colors, defined in CheckersConstants
	private int[] playerColors;

	// turn variable
	private int currentTurn;

	// is there a game going on?
	private boolean gameActive;

	// the model that holds the pieces
	private Board theBoard;

	// general message sent to the gui when informing is necessary
	private String message;

	// currently selected space
	private int selectedIndex;
	
	// necessary to see which player this game is when networked
	private int playerNum = -1;

	/**
	 * Constructor
	 * 
	 * Create the driver, which in turn creates the rest of the system.
	 */
	public CheckersGame(Board b) {
		// Create the board
		theBoard = b;

		// initialize currentTurn to player 0
		currentTurn = 0;

		// start the game as "stopped"
		gameActive = false;

		// initialize selectedIndex to "not selected"
		selectedIndex = -1;

		// playerName initialization
		numPlayers = 2;
		playerNames = new String[numPlayers];
		playerNames[0] = "Player1";
		playerNames[1] = "Player2";

		playerColors = new int[numPlayers];
		selectColors();
	}

	/**
	 * This method is called after a move has been checked. Changes active
	 * player when a final succesful jump has been made, resets the timer when
	 * appropriate, and tells the appropriate player whos turn it is to make a
	 * move.
	 * 
	 * Refactored: The method should be called when the turn is over. Checking
	 * should be completed outside of this method. It will still perform all
	 * operations needed at end of turn.
	 * 
	 * @param color
	 *            The color whose turn it will now be
	 * @param space
	 *            The space on the board from which a multiple jump has to be
	 *            made
	 * 
	 * @pre a players has made a move
	 * @post a player has been told to make a move
	 */
	public void endTurn() {

		this.currentTurn = (3 - currentTurn) % 2; // will change currentTurn
		// to opposite
		// resetTimer();
		checkEnd();
		setChanged();
		notifyObservers();
	}

	/**
	 * This method ends the checkers game due to whatever reason neccessary ie.
	 * a draw, someone quitting, or a victory.
	 * 
	 * @param message
	 *            the message to send to all players regarding the reason for
	 *            ending the game
	 * 
	 * @pre the criteria for ending a game has been met, depending on why the
	 *      game ended
	 * @post the game has been ended for both players and the game is ready to
	 *       exit
	 */
	public void endGame(Object msg) {

		gameActive = false;
		setChanged();
		notifyObservers(msg);
	}

	/**
	 * This method ends the game in a draw, alerting both players that the draw
	 * has taken place
	 * 
	 * @pre both players have agreed to a draw
	 * @post the game has ended and both players have been notified of the draw
	 */
	public void endInDraw(int playerNum) {
		message = "Draw accepted. Game over.";
		endGame(ACCEPT);
	}

	/**
	 * Ends the game as a result of a player quitting, notifying each player
	 * 
	 * @param the
	 *            player who quit
	 */
	public void endInQuit(int playerNum) {
		message = playerNames[playerNum] + " has resigned from the game.";
		endGame(RESIGN);
	}
	
	/**
	 * This method sets the colors of pieces that each player will be
	 * 
	 * @pre the game has been started, and there are 2 players
	 * @post each player has their colors
	 */
	public void selectColors() {
		// Randomly select color for each player
		playerColors[0] = (int) (Math.random() * 100.0 % 2);
		playerColors[1] = (3 - playerColors[0]) % 2;
	}

	/**
	 * This method will start the game play. Letting the first person move their
	 * piece and so on
	 * 
	 * @pre There are 2 players to play, and all pregame conditions are in place
	 * @post The first person is able to make their first move
	 */
	public void startGame() {
		gameActive = true;
		setChanged();
		notifyObservers();
	}

	/**
	 * Determines if a space is selectable based on whose turn it is. If a space
	 * has already been selected, calls movePiece to perform a move
	 * 
	 * @param space
	 *            the space to select
	 * @return True on select or move, false otherwise
	 */
	public boolean selectSpace(int space) {
		boolean retval = false; // was selection successful
		if (gameActive){
			
			if (theBoard.occupied(selectedIndex)) {
				Move newMove = new Move(selectedIndex, space);
				movePiece(newMove);
				
				selectedIndex = -1;
				retval = true;
			} else if (theBoard.occupied(space)) {
				//checks to see if the player's piece is selected
				if (theBoard.getPieceAt(space).getColor() == getPlayerColor(currentTurn)) {
					selectedIndex = space;
					retval = true;
				}
			}
			setChanged();
			notifyObservers();
			
		}
		return retval;
	}

	/**
	 * @param newMove
	 */
	private void movePiece(Move newMove) {
		if (newMove.validate()) {			
			// notify network
			setChanged();
			notifyObservers(newMove);
			
			boolean isJump = newMove.isJump();
			
			theBoard.movePiece(newMove);
			// if the piece is at one of the end rows, and is not a king, make
			// it a king
			if ((newMove.endLocation() < 4 || newMove.endLocation() > 27)
					&& theBoard.getPieceAt(newMove.endLocation()).getType() == SINGLE) {
				theBoard.kingPiece(newMove.endLocation());
			}
			
			if (isJump){
				theBoard.removePiece(newMove.getJumpIndex());
				
				// if we're playing with forced jumps
				if (Rules.getInstance() instanceof ForceJumpValidator){
					// check to see if there are any forced jumps, if not, end the turn.
					if ( !((ForceJumpValidator)Rules.getInstance()).pieceHasForceJumps(newMove.endLocation()) ){
						endTurn();
					}
				}else{					
					endTurn();
				}
			}else{
				endTurn();
			}
		}
	}
	
	/**
	 * Checks to see if the player who is about to start their turn has an pieces left 
	 * to move.  If not, set gameActive to false.
	 *
	 */
	private void checkEnd() {
		int stillPlay = 0;
		for(int i = 0; i < NUM_SPACES; i++) {
			Piece curPiece = theBoard.getPieceAt(i);
			if(curPiece != null){
				if(curPiece.getColor() == getPlayerColor(currentTurn)){
					stillPlay++;
				}
			}
		}
		if(stillPlay == 0){
			gameActive = false;
		}
	}
	
	/**
	 * Offer a draw
	 */
	public void offerDraw() {
		setChanged();
		notifyObservers(DRAW);
		
	}

	/**
	 * Returns the name of player with number passed in
	 * 
	 * @param playerNumber
	 *            Number of player to get name for
	 * 
	 * @return Player's name of number passed in
	 */
	public String getPlayerName(int playerNumber) {
		return playerNames[playerNumber];
	}

	/**
	 * Set the name for the player using the passed in values.
	 * 
	 * @param num
	 *            The player's number (1 or 2)
	 * @param name
	 *            The name to assign to the player.
	 */
	public void setPlayerName(int num, String name) {
		playerNames[num] = name;
		setChanged();
		notifyObservers();
	}

	/**
	 * Returns color for player number passed in
	 * 
	 * @param playerNumber
	 *            Number of player to get color for
	 * 
	 * @return Color for player number passed in
	 */
	public int getPlayerColor(int playerNumber) {
		return playerColors[playerNumber];
	}

	/**
	 * Set the color for a player using the passed in value.
	 * 
	 * @param num
	 *            The player's number (1 or 2)
	 * @param color
	 *            The color to assign to the player.
	 */
	public void setPlayer1Color(int color) {
		playerColors[0] = color;
		playerColors[1] = (3 - color) % 2;		
	}

	/**
	 * @return Returns the color whose turn it is.
	 */
	public int getCurrentTurn() {
		return currentTurn;
	}

	/**
	 * @return Returns the gameActive.
	 */
	public boolean isGameActive() {
		return gameActive;
	}

	/**
	 * @return Returns the message.
	 */
	public String getMessage() {
		return message;
	}

	/**
	 * @return Returns the selectedIndex.
	 */
	public int getSelectedIndex() {
		return selectedIndex;
	}

	/**
	 * @return Returns the playerNum.
	 */
	public int getPlayerNum() {
		return playerNum;
	}

	/**
	 * @param playerNum The playerNum to set.
	 */
	public void setPlayerNum(int playerNum) {
		this.playerNum = playerNum;
	}

}// CheckersGame.java
