package checkers.model;

/**
 * ForceJumpValidator.java
 * 
 * Version:
 * 		$Id: ForceJumpValidator.java,v 1.7 2005/11/06 22:48:06 mtg9625 Exp $
 * 
 * Revisions:
 * 		$Log: ForceJumpValidator.java,v $
 * 		Revision 1.7  2005/11/06 22:48:06  mtg9625
 * 		Fixed forced jumps
 * 		
 * 		Revision 1.6  2005/11/05 19:09:53  jim5870
 * 		fixed
 * 		
 * 		Revision 1.5  2005/11/05 18:16:53  jim5870
 * 		made pieceHasForceJumps public and return boolean
 * 		
 * 		Revision 1.4  2005/11/05 17:36:38  jim5870
 * 		updated pieces constants to use CheckersConstants
 * 		
 * 		Revision 1.3  2005/11/05 17:23:32  jim5870
 * 		comments added
 * 		fixed null pointer exception and final validation step
 * 		
 */

import java.util.LinkedList;
import java.util.ListIterator;
import checkers.CheckersConstants;

/**
 * This class is used to check for force jumps. In the case where force jumps
 * exist the move provided is checked to make sure that it is a force jump.
 */

public class ForceJumpValidator extends Rules implements CheckersConstants{

	/* Due to possibility of one start position having multiple force jumps,
	 * the start positions are kept in one list and the same index in the end
	 * positions list corresponds to the end position of the force jump */
	private LinkedList myForceStarts;	// holds the start positions of force jumps
	private LinkedList myForceEnds;		// holds the end positions of force jumps
	
	/**
	 * Constructor
	 */
	public ForceJumpValidator(){
	}
	
	/**
	 * validateMove
	 * 
	 * Validates that the move is a force jump if force jumps exist. If no
	 * force jumps exist the move is validated normally.
	 * 
	 * @param	move	the move to be validated
	 * @return	returns true if the move is valid
	 */
	public boolean validateMove(Move move){
		boolean retVal = false;					// default return value
		myForceStarts = new LinkedList();		// need empty lists for each validation
		myForceEnds = new LinkedList();		
		
		// check for any force jumps on the board
		if(containsForceJump(move.startLocation())){
			// check that the move is an existing force jump
			retVal = isForceJump(move);
		}
		else{
			// validate the move normally
			retVal = super.validateMove(move);
		}
		return retVal;
	}
	
	/**
	 * isForceJump
	 * 
	 * Since the board contains force jumps the move has to be validated as one
	 * of the existing force jumps.
	 * 
	 * @param 	move	the move to be validated
	 * @return	returns true if the move is an existing force jump
	 */
	private boolean isForceJump(Move move){
		boolean retVal = false;				// default return value
		int start = move.startLocation();	// starting index of the move
		int end = move.endLocation();		// ending index of the move
		int startIndex;						// position in list to start checking
		if((startIndex = myForceStarts.indexOf(new Integer(start))) != -1){
			// iterates through the list to check that move is an existing force jump
			retVal = checkForceJump(start,end,startIndex);
		}
		return retVal;
	}
	
	/**
	 * containsForceJump
	 * 
	 * Checks whether or not there are existing force jumps on the board
	 * 
	 * @param 	start	the position of the starting piece - only needed to get color
	 * @return	returns true if the board contains force jumps
	 */
	private boolean containsForceJump(int start){
		boolean retVal = false;		// default return value
		int color = myBoard.getPieceAt(start).getColor();	// color of piece to move
		
		// go through the board and check each colored piece for force jumps
		for(int i=0;i<myBoard.sizeOf();i++){
			if(myBoard.getPieceAt(i) != null){
				// only check pieces of a certain color
				if(myBoard.getPieceAt(i).getColor() == color){
					// check to see if the piece has any force jumps
					pieceHasForceJumps(i);
				}
			}
		}
		
		// set return value based on whether force jumps were added to the collection
		retVal = myForceStarts.size() > 0;
		return retVal;
	}
	
	/**
	 * pieceHasForceJumps
	 * 
	 * Check whether a piece has any force jumps. If it does, store the start position
	 * in the collection along with its corresponding end position in the other list
	 * 
	 * @param 	pos		the position on the board that needs to be checked
	 */
	public boolean pieceHasForceJumps(int pos){
		boolean retVal = false;
		Piece myPiece = myBoard.getPieceAt(pos);	// the piece at the position
		
		// check if the piece is blue or a white king
		if((myPiece.getColor() == BLUE) ||
			(myPiece.getColor() == WHITE && myPiece.getType() == KING)){
			// check for possible valid jump and add positions to start and end lists
			if((!myBoard.occupied(pos+7) && super.isJump(pos,pos+7))){
				myForceStarts.add(new Integer(pos));
				myForceEnds.add(new Integer(pos+7));
				retVal = true;
				
			}
			if((!myBoard.occupied(pos+9) && super.isJump(pos,pos+9))){
				myForceStarts.add(new Integer(pos));
				myForceEnds.add(new Integer(pos+9));
				retVal = true;
			}	
		}
		
		// check if the piece is white or a blue king
		if((myPiece.getColor() == WHITE) ||
			(myPiece.getColor() == BLUE && myPiece.getColor() == KING)){
			// check for possible valid jump and add positions to start and end lists
			if((!myBoard.occupied(pos-7) && super.isJump(pos,pos-7))){
				myForceStarts.add(new Integer(pos));
				myForceEnds.add(new Integer(pos-7));
				retVal = true;
				
			}
			if((!myBoard.occupied(pos-9) && super.isJump(pos,pos-9))){
				myForceStarts.add(new Integer(pos));
				myForceEnds.add(new Integer(pos-9));
				retVal = true;
			}
		}
		return retVal;
	}
	
	/**
	 * checkForceJump
	 * 
	 * Checks to see whether or not the move provided is an existing force jump
	 * 
	 * @param 	start 	the start position of the given move
	 * @param 	end		the end position of the given move
	 * @param 	startIndex	the position to start iterating at in the force jump lists
	 * @return	returns true if the move is also an existing force jump
	 */
	private boolean checkForceJump(int start,int end,int startIndex){
		boolean retVal = false;		// default return value
		ListIterator startItr = myForceStarts.listIterator(startIndex); // start positions list iterator
		ListIterator endItr = myForceEnds.listIterator(startIndex); // start positions list iterator
		
		// iterate through the list 
		while(startItr.hasNext() && !retVal){
			// get list values
			Integer forceStart = (Integer)startItr.next();
			Integer endStart = (Integer)endItr.next();
			// check that the start and end of the move equal the start and end of a force jump 
			retVal = (forceStart.equals(new Integer(start))) && (endStart.equals(new Integer(end)));
		}
		return retVal;
	}
}
