package checkers.model;
/**
 * Rules.java
 *
 * Version:
 *    $Id: Rules.java,v 1.7 2005/11/05 18:30:40 jim5870 Exp $
 *
 * Revisions:
 *    $Log: Rules.java,v $
 *    Revision 1.7  2005/11/05 18:30:40  jim5870
 *    fixed another null exception
 *
 *    Revision 1.6  2005/11/05 18:03:45  jim5870
 *    checks for null pieces
 *
 *    Revision 1.5  2005/11/05 17:36:25  jim5870
 *    updated pieces constants to use CheckersConstants
 *
 *    Revision 1.4  2005/11/05 17:22:50  jim5870
 *    comments added
 *
 *    Revision 1.3  2005/11/05 15:29:35  pjk7060
 *    fixess
 *
 *    Revision 1.2  2005/11/05 15:12:35  jim5870
 *    refactored rules class
 *
 *    Revision 1.1  2005/11/02 04:34:05  pjk7060
 *    moved classes into the proper packages
 *
 *    Revision 1.1  2005/11/02 04:31:20  pjk7060
 *    split everything into packages
 *
 *    implemented the networking class
 *
 *    Revision 1.3  2005/10/29 14:22:31  jmr1040
 *    Fixed formatting (so we can see all nesting)
 *
 *    Revision 1.2  2005/10/29 14:16:12  jmr1040
 *    Renamed Driver and Facade to match purpose
 *
 *    Revision 1.1  2005/10/20 13:53:36  jmr1040
 *    Initial state of Checkers project
 *
 *    Revision 1.1  2002/10/22 21:12:53  se362
 *    Initial creation of case study
 *
 */

import checkers.CheckersConstants;

/**
 * This class is used to check the validity of the moves made by the players. It
 * also checks to see if the conditions for the end of the game have been met.
 * 
 * @author
 * @author
 * 
 */
public class Rules implements CheckersConstants{

	protected Board myBoard; 			// The board
	protected int myJumpIndex;			// the index of the piece to be jumped
	protected static Rules myInstance = null;	// singelton instance
	
	/**
	 * Constructor
	 */
	protected Rules(){		
	}
	
	/**
	 * getInstance
	 * 
	 * Creates the singleton instance of the Rules object
	 * 
	 * @return the Rules object
	 */
	public static Rules getInstance(){
		// Creates the instance if it hasn't been created already
		if(myInstance == null){
			myInstance = new ForceJumpValidator();
		}
		return myInstance;
	}
	
	/**
	 * setBoard
	 * 
	 * Sets the board
	 * 
	 * @param theBoard the checker board
	 */
	public void setBoard(Board theBoard) {
		myBoard = theBoard;
	}

	/**
	 * This method checks to see if the move that was just made was valid and
	 * returns a boolean indicating that.
	 * 
	 * @param move
	 *            The move that is to be validated.
	 * 
	 * @return Boolean indicating if it was valid.
	 */
	public boolean validateMove(Move move) {
		boolean retVal = false;				// default return value
		int start = move.startLocation();	// start index of move
		int end = move.endLocation();		// end index of move
		
		// checks to see that the end index selected isn't occupied
		if(!myBoard.occupied(end)){
			// checks that the move is a valid adjacent move
			if(!(retVal = isAdjacent(start,end))){
				// if not adjacent, checks that the move is a valid jump
				retVal = (isJump(start,end));
			}
		}
		return retVal;	
	}

	/**
	 * isAdjacent
	 * 
	 * Checks the move to see if it is a valid adjacent move.
	 * 
	 * @param 	start 	the starting position index
	 * @param 	end		the ending position index
	 * @return	returns true if the move is a valid adjacent move
	 */
	private boolean isAdjacent(int start, int end){
		boolean retVal = false;		// default return value
		int rowStart = start/4;		// get the row number of the start position
		int rowEnd = end/4;			// get the row number of the end position
		
		// check if start and end rows are at least one row apart up or down
		if(Math.abs(rowStart-rowEnd) == 1){			
			// check valid direction of piece
			if(isValidDirection(rowStart,rowEnd,start)){
				int diff = end - start;		// the numerical difference between start and end positions
				
				// for adjacent moves the board indices for valid moves 
				// differ by one dependiong on an odd or even row
				if(rowStart%2 == 0){
					retVal = adjacentEvenRowMove(rowStart,rowEnd,diff);
				}
				else{
					retVal = adjacentOddRowMove(rowStart,rowEnd,diff);
				}
			}
		}
		return retVal;
	}
	
	/**
	 * isJump
	 * 
	 * Checks to see if the move is a valid jump
	 * 
	 * @param start the start position of the given move
	 * @param end the end position of the given move
	 * @return returns true if the move is a valid jump
	 */
	public boolean isJump(int start, int end){
		boolean retVal = false;		// default return value
		int rowStart = start/4;		// starting row
		int rowEnd = end/4;			// ending row
		
		// make sure the move spans two rows and the end position is within the 
		// bounds of the board
		if(Math.abs(rowStart-rowEnd) == 2 && end >= 0 && end < myBoard.sizeOf()){
			// validate direction of the move
			if(isValidDirection(rowStart,rowEnd,start)){
				int diff = end - start;	// numerical difference between positions
				// the actual starting and end position differences are the same,
				// but the index of the jumped piece differs according to odd or even row
				if(rowStart%2 == 0){
					retVal = jumpEvenRowMove(rowStart,rowEnd,start,diff);
				}
				else{
					retVal = jumpOddRowMove(rowStart,rowEnd,start,diff);
				}
			}
		}
		return retVal;
	}
	
	/**
	 * isValidDirection
	 * 
	 * Checks to see that the move is in a valid direction according to the type and color
	 * 
	 * @param rowStart 	the starting row
	 * @param rowEnd   	the end row
	 * @param start	   	the index of the piece selected to move
	 * @return			returns true if the direction is valid
	 */
	private boolean isValidDirection(int rowStart,int rowEnd,int start){
		boolean retVal = false;	// default return value

		if(myBoard.getPieceAt(start) != null){
			// if the piece is a king any direction is valid
			if(!(retVal = myBoard.getPieceAt(start).getType() == KING)){
				// check for blue piece moving from low to high
				if(myBoard.getPieceAt(start).getColor() == BLUE){
					retVal = (rowEnd > rowStart);
				}
				// else check for white piece moving from high to low
				else{
					retVal = (rowStart > rowEnd);
				}
			}
		}
		return retVal;
	}
	
	/**
	 * adjacentEvenRowMove
	 * 
	 * Checks to see if the adjecent move from an even row is valid
	 * 
	 * @param rowStart 	the starting row
	 * @param rowEnd	the ending row
	 * @param diff		the numerical difference between the start and end positions
	 * @return	returns true if the adjacent move is valid
	 */
	private boolean adjacentEvenRowMove(int rowStart,int rowEnd,int diff){
		boolean retVal = false;		// default return value
		// check to see if direction is low to high
		if(rowEnd > rowStart){
			// varifies differences in start and end positions
			retVal = (diff == 3) || (diff == 4);
		}
		else{
			// varifies differences in start and end positions
			retVal = (-diff == 4) || (-diff == 5);
		}
		return retVal;
	}
	
	/**
	 * adjacentOddRowMove
	 * 
	 * Checks to see if the adjacent move from an odd row is valid
	 * 
	 * @param rowStart 	the starting row
	 * @param rowEnd	the ending row
	 * @param diff		the numerical difference between the start and end positions
	 * @return returns true if the move is a valid adjacent move
	 */
	private boolean adjacentOddRowMove(int rowStart,int rowEnd,int diff){
		boolean retVal = false;		// default return value
		// check to see if direction is low to high
		if(rowEnd > rowStart){
			// varifies differences in start and end positions
			retVal = (diff == 4) || (diff == 5);
		}
		else{
			// varifies differences in start and end positions
			retVal = (-diff == 3) || (-diff == 4);
		}
		return retVal;
	}
	
	/**
	 * jumpOddRowMove
	 * 
	 * Checks to see if the jump from an odd row is valid
	 * 
	 * @param rowStart 	the starting row
	 * @param rowEnd	the ending row
	 * @param start		the starting index
	 * @param diff		the numerical differences
	 * @return
	 */
	private boolean jumpOddRowMove(int rowStart,int rowEnd, int start, int diff){
		boolean retVal = false;		// default return value
		int jumpIndex = -1;			// default jump index
		
		// check to see if moving from low to high
		if(rowEnd > rowStart){
			if(diff == 7){
				// check that piece being jumped is opposite color and store jump index
				if(myBoard.getPieceAt(start+4)!= null){
					retVal = myBoard.getPieceAt(start).getColor() != myBoard.getPieceAt(start+4).getColor();				
					jumpIndex = start+4;
				}
			}
			else if(diff == 9){
				// check that piece being jumped is opposite color and store jump index
				if(myBoard.getPieceAt(start+5)!= null){
					retVal = myBoard.getPieceAt(start).getColor() != myBoard.getPieceAt(start+5).getColor();
					jumpIndex = start+5;
				}
			}
		}
		else{
			if(-diff == 7){
				// check that piece being jumped is opposite color and store jump index
				if(myBoard.getPieceAt(start-3)!= null){
					retVal = myBoard.getPieceAt(start).getColor() != myBoard.getPieceAt(start-3).getColor();
					jumpIndex = start-3;
				}
			}
			else if(-diff == 9){
				// check that piece being jumped is opposite color and store jump index
				if(myBoard.getPieceAt(start-4)!= null){
					retVal = myBoard.getPieceAt(start).getColor() != myBoard.getPieceAt(start-4).getColor();
					jumpIndex = start-4;
				}
			}
		}
		// if the piece is a valid jump set the jumped piece index
		if(retVal){
			setJumpIndex(jumpIndex);
		}
		return retVal;
	}
	
	/**
	 * jumpEvenRowMove
	 * 
	 * Checks to see if the move is a valid jump from an even row
	 * 
	 * @param rowStart	the starting row
	 * @param rowEnd	the ending row
	 * @param start		the index of the starting position
	 * @param diff		the numerical difference between the starting and ending positions
	 * @return	returns true if the move is a valid jump from an even row
	 */
	private boolean jumpEvenRowMove(int rowStart,int rowEnd, int start, int diff){
		boolean retVal = false;		// default return value
		int jumpIndex = -1;			// default jump index value

		// check to see if moving from low to high
		if(rowEnd > rowStart){
			if(diff == 7){
				// check that piece being jumped is opposite color and store jump index
				if(myBoard.getPieceAt(start+3)!= null){
					retVal = myBoard.getPieceAt(start).getColor() != myBoard.getPieceAt(start+3).getColor();
					jumpIndex = start+3;	
				}
			}
			else if(diff == 9){
				// check that piece being jumped is opposite color and store jump index
				if(myBoard.getPieceAt(start+4)!= null){
					retVal = myBoard.getPieceAt(start).getColor() != myBoard.getPieceAt(start+4).getColor();
					jumpIndex = start+4;
				}
			}
		}else{
			if(-diff == 7){
				// check that piece being jumped is opposite color and store jump index
				if(myBoard.getPieceAt(start-4)!= null){
					retVal = myBoard.getPieceAt(start).getColor() != myBoard.getPieceAt(start-4).getColor();
					jumpIndex = start-4;
				}
			}
			else if(-diff == 9){
				// check that piece being jumped is opposite color and store jump index
				if(myBoard.getPieceAt(start-5)!= null){
					retVal = myBoard.getPieceAt(start).getColor() != myBoard.getPieceAt(start-5).getColor();
					jumpIndex = start-5;
				}
			}
		}
		// if the piece is a valid jump set the jumped piece index
		if(retVal){
			setJumpIndex(jumpIndex);
		}
		return retVal;
	}
	
	/**
	 * setJumpIndex
	 * 
	 * set the index of the jumped piece
	 * 
	 * @param index	the jumped piece index
	 */
	private void setJumpIndex(int index){
		myJumpIndex = index;
	}
	
	/**
	 * getJumpIndex
	 * 
	 * get the index of the jumped piece
	 * 
	 * @return the index of the jumped piece
	 */
	public int getJumpIndex(){
		return myJumpIndex;
	}

}// Rules.java
