package checkers.view;
/*
 * CheckerGUI.java
 * 
 * The actual board.
 *
 * Created on January 25, 2002, 2:34 PM
 * 
 * Version
 * $Id: CheckerGUI.java,v 1.22 2005/11/08 23:02:10 pjk7060 Exp $
 * 
 * Revisions
 * $Log: CheckerGUI.java,v $
 * Revision 1.22  2005/11/08 23:02:10  pjk7060
 * comments
 *
 * Revision 1.21  2005/11/08 22:37:00  pjk7060
 * removed unused timer code
 *
 * fixed warnings
 *
 * Revision 1.20  2005/11/07 05:05:02  mtg9625
 * disable draw and resign button when it's not your turn,
 * mainly important for network play
 *
 * Revision 1.19  2005/11/07 00:01:30  mtg9625
 * Draw functionality implemented, at least locally
 *
 * Revision 1.18  2005/11/06 23:16:05  mtg9625
 * Added methods to deal with a draw
 *
 * Revision 1.16  2005/11/06 22:48:06  mtg9625
 * Fixed forced jumps
 *
 * Revision 1.15  2005/11/06 22:14:24  mtg9625
 * Checks if a player has won and displays message
 * Resign button works and ends game
 *
 * Revision 1.14  2005/11/06 20:18:54  pjk7060
 * Fixed turn issues
 *
 * Revision 1.13  2005/11/06 19:38:46  pjk7060
 * worked many bugs out of networking
 *
 * Revision 1.12  2005/11/05 19:29:40  pjk7060
 * messages
 *
 * Revision 1.11  2005/11/05 19:18:34  pjk7060
 * methodized update
 *
 * Revision 1.10  2005/11/05 18:42:19  pjk7060
 * fixed jumps
 *
 * Revision 1.9  2005/11/05 18:03:37  pjk7060
 * should remove piece
 *
 * Revision 1.8  2005/11/05 17:40:40  pjk7060
 * working on update, displays pieces and current turn
 *
 * Revision 1.7  2005/11/05 15:35:32  pjk7060
 * Little fixes
 *
 * Revision 1.6  2005/11/04 20:04:34  mtg9625
 * now displays completely, still need to do update method
 * fixed outofbounds error in board, (added a - 1)
 * fixed nullPointerException in CheckersGame(didn't initialize array and tried to access)
 * removed imports of deleted classes in GameController
 *
 * Revision 1.5  2005/11/04 16:22:04  mtg9625
 * changed buttons to an array list, only 32
 * one spacing button, does nothing
 * split frame into panels
 * now accepts an outside actionlistener
 * so far board displays correctly, working on other panels
 *
 * Revision 1.4  2005/11/04 15:50:16  pjk7060
 * Fixed registering of components with proper observers, and proper references
 *
 * Revision 1.3  2005/11/04 04:05:37  pjk7060
 * Created Global Constants interface
 *
 * Further development on controller, and initial game setup
 *
 * Revision 1.2  2005/11/02 04:34:05  pjk7060
 * moved classes into the proper packages
 *
 * Revision 1.1  2005/11/02 04:31:21  pjk7060
 * split everything into packages
 *
 * implemented the networking class
 *
 * Revision 1.3  2005/10/29 14:22:31  jmr1040
 * Fixed formatting (so we can see all nesting)
 *
 * Revision 1.2  2005/10/29 14:16:12  jmr1040
 * Renamed Driver and Facade to match purpose
 *
 * Revision 1.1  2005/10/20 13:53:36  jmr1040
 * Initial state of Checkers project
 *
 * Revision 1.1  2002/10/22 21:12:52  se362
 * Initial creation of case study
 *
 */


import java.awt.*;
import java.awt.event.ActionListener;
import java.awt.event.WindowListener;
import java.util.*;
import javax.swing.*;
import javax.swing.border.BevelBorder;

import checkers.CheckersConstants;
import checkers.model.Board;
import checkers.model.CheckersGame;


/**
 * 
 * @author
 * @version
 */

public class CheckerGUI extends JFrame implements CheckersConstants, Observer {

	// eclipse stuff
	private static final long serialVersionUID = 1L;

	//Warning for when low on time
	private static final String warning = "In Warning Time.";
	
	//The board for the game
	private Board board;
	
	//The panel the holds the board
	private JPanel boardPanel;
	
	//The Panel that holds the name of the player on top
	private JPanel topPanel;
	
	//the panel that holds the name of the player on bottm
	private JPanel bottomPanel;
	
	//the panel that holds extra buttons, and whosTurn
	private JPanel infoPanel;

	//the squares in the game that can have pieces on them
	private ArrayList squares;
	
	//the label that goes on top
	private JLabel playerOneLabel;

	//the label that goes on bottom
	private JLabel playerTwoLabel;

	//the label for the time remaining for the current turn
	private JLabel timeRemainingLabel;

	//the button to resign the game
	private JButton resignButton;

	//the button to offer a draw
	private JButton drawButton;

	//used to warn of low time
	private JLabel warningLabel; 
	
	//holds the name of the player whos turn it is
	private JLabel whosTurnLabel;
	
	//listens to events in the frame
	private ActionListener listener;
	
	//true if the players names have been set
	private boolean namesSet;
	
	//if the listener has been added
	private boolean actionListenerAdded = true;
	
	//no show, used to accept a draw
	private JButton acceptDraw;


	/**
	 * 
	 * Constructor, creates the GUI and all its components
	 * 
	 * @param facade
	 *            the facade for the GUI to interact with
	 * @param name1
	 *            the first players name
	 * @param name2
	 *            the second players name
	 * 
	 */

	public CheckerGUI(ActionListener listener, Board board) {

		super("Checkers");
		
		this.listener = listener;
		this.board = board;
		
		squares = new ArrayList(NUM_SPACES);
		namesSet = false;
		
		addWindowListener((WindowListener)listener);

		initComponents(listener);
	}


	/**
	 * This method initializes all the components of the GUI
	 * Giving them there values, and setting necessary preferences
	 * 
	 */
	private void initComponents(ActionListener listener) {

	
		//Make the window not resizeable
		this.setResizable(false);
		
		//give the window a borderlayout
		this.setLayout(new BorderLayout());
		
		//The current button being added
		JButton current;

		//Create all the buttons, setting color, actioncommands, and actionlistener
		for(int i = 0; i < NUM_SPACES; i++) {
			squares.add(i,new JButton());
			current = ((JButton)squares.get(i));
			current.setActionCommand(Integer.toString(i));
			current.addActionListener(listener);
			current.setBackground(Color.WHITE);
			current.setBorder(BorderFactory.createEtchedBorder(Color.BLACK,Color.DARK_GRAY));
			current.setPreferredSize(new Dimension(80,80));
		}

		playerOneLabel = new JLabel();
		playerOneLabel.setPreferredSize(new Dimension(160,30));
		playerTwoLabel = new JLabel();
		playerTwoLabel.setPreferredSize(new Dimension(160,30));
		whosTurnLabel = new JLabel();
		whosTurnLabel.setBorder(BorderFactory.createBevelBorder(BevelBorder.LOWERED));
		whosTurnLabel.setPreferredSize(new Dimension(80,30));

		warningLabel = new JLabel();
		warningLabel.setText(warning);
		warningLabel.setVisible(false);
		
		timeRemainingLabel = new JLabel("NO TIME");
		timeRemainingLabel.setPreferredSize(new Dimension(75,27));
		timeRemainingLabel.setBorder(BorderFactory.createBevelBorder(BevelBorder.LOWERED));

		resignButton = new JButton("Resign");
		resignButton.setActionCommand(RESIGN);
		resignButton.addActionListener(listener);
		resignButton.setPreferredSize(new Dimension(75, 27));

		drawButton = new JButton("Draw");
		drawButton.setActionCommand(DRAW);
		drawButton.addActionListener(listener);
		drawButton.setPreferredSize(new Dimension(75,27));
		
		acceptDraw = new JButton();
		acceptDraw.setActionCommand(ACCEPT);
		acceptDraw.addActionListener(listener);
		
		//init the separare panels for the screen
		//panel for the board
		boardPanel = new JPanel();
		//panel that holds the name of the player on the top of the screen
		topPanel = new JPanel();
		//panel that holds the name of the player on the bottom of the screen
		bottomPanel = new JPanel();
		//panel that holds time, who's turn, draw and resign buttons
		infoPanel = new JPanel();
		
		buildScreen();
	}//InitComponents
	
	private void buildScreen() {

		//Build the boardPanel
		
		boardPanel.setLayout(new GridBagLayout());
		boardPanel.setBackground(Color.RED);
		//used for the playable buttons
		GridBagConstraints constraints1 = new GridBagConstraints();
		//used for the space button
		GridBagConstraints constraints2 = new GridBagConstraints();
		
		//Button used for spaceing on the board, not clickable
		JLabel space = new JLabel();
		space.setEnabled(false);
		space.setBackground(Color.RED);
		space.setPreferredSize(new Dimension(80,80));
		
		//track the row being used
		constraints1.gridy = 0;
		constraints2.gridy = 0;
		
		for(int i = 0; i < NUM_SPACES; i++) {
			if(constraints1.gridy%2 == 0) {
				constraints1.gridx = 2*(i%4);
				constraints2.gridx = 2*(i%4) + 1;
			} else {
				constraints1.gridx = 2*(i%4) + 1;
				constraints2.gridx = 2*(i%4);
			}
			boardPanel.add((JButton)squares.get(i), constraints1);
			boardPanel.add(space, constraints2);
			if(i%4 == 3){
				constraints1.gridy += 1;
				constraints2.gridy += 1;
			}
		}
		boardPanel.setBorder(BorderFactory.createBevelBorder(BevelBorder.RAISED, Color.WHITE,Color.BLACK));
		
		//Build the infoPanel, some variables reused
		infoPanel.setLayout(new GridBagLayout());
		constraints1.insets = new Insets(5,5,5,5);
		constraints1.gridy = 0;
		infoPanel.add(new JLabel("Turn:"), constraints1);
		constraints1.gridy = 1;
		infoPanel.add(whosTurnLabel, constraints1);
		constraints1.gridy = 2;
		infoPanel.add(drawButton, constraints1);
		constraints1.gridy = 3;
		infoPanel.add(resignButton, constraints1);
		constraints1.gridy = 4;
		infoPanel.add(new JLabel("Time Remaining:"), constraints1);
		constraints1.gridy = 5;
		infoPanel.add(timeRemainingLabel, constraints1);
		
		
		//Build the topPanel
		topPanel.setLayout(new GridBagLayout());
		topPanel.add(playerOneLabel);
		
		//Build the bottomPanel
		bottomPanel.setLayout(new GridBagLayout());
		bottomPanel.add(playerTwoLabel);
	
		setPieces();
		
		//Add the panels to the frame
		getContentPane().add(boardPanel, BorderLayout.CENTER);
		getContentPane().add(infoPanel, BorderLayout.EAST);
		getContentPane().add(topPanel, BorderLayout.NORTH);
		getContentPane().add(bottomPanel, BorderLayout.SOUTH);
		pack();
	}//BuildScreen

	/**
	 * Updates the GUI reading the pieces in the board Puts pieces in correct
	 * spaces, updates whos turn it is, updates time, displays necessary popups
	 * 
	 * @param the
	 *            board
	 */


	public void update(Observable arg, Object obj) {
		if(arg instanceof CheckersGame) {
			
			//saves from having to cast to game at every use
			CheckersGame game = (CheckersGame)arg;
			
			//Set the player names
			namesSet = setNames(game);
			
			//Set the label to say whos turn it is
			whosTurnLabel.setText(game.getPlayerName(game.getCurrentTurn()));
			
			//See if a message was sent
			if(obj instanceof String) {
				//Check if it was a resign message
				if(((String)obj).equals(RESIGN)){
					resignGame(game.getMessage(), game.getPlayerName((game.getCurrentTurn() + 1)%2));
				}else if(((String)obj).equals(DRAW)){
					if(offerDraw()){
						acceptDraw.doClick();
					}
				}else if(((String)obj).equals(ACCEPT)){
					acceptDraw();
				}
				//Check if there is a winner
			} else 	if(!game.isGameActive() && namesSet){
				showWin(game.getPlayerName((game.getCurrentTurn() + 1)%2));
			}
			//Highlight the selected square
			if(game.getSelectedIndex() > -1){
				((JButton)squares.get(game.getSelectedIndex())).setBackground(Color.GREEN);
			} else{
				for(int i = 0; i < NUM_SPACES; i++){
					((JButton)squares.get(i)).setBackground(Color.WHITE);
				}
			}
			
			//Used mainly for networking, disables the opponents pieces
			setTurn(game.getCurrentTurn() == game.getPlayerNum() || game.getPlayerNum() == -1);
			
			//Redraw the pieces to show the move
			setPieces();
		}
		
		if(arg instanceof Board){			
			setPieces();
		}
	}
	
	private boolean setNames(CheckersGame game){
		//if the names are already set
		boolean set = false;
		
		//if the names aren't set
		if(!namesSet) {
			//if the top label's text doesn't match the blue player's label, set it
			if(playerOneLabel.getText() != game.getPlayerName(game.getPlayerColor(BLUE))
			|| playerTwoLabel.getText() != game.getPlayerName(game.getPlayerColor(WHITE))){
				
				playerOneLabel.setText(game.getPlayerName(game.getPlayerColor(BLUE)));
				playerTwoLabel.setText(game.getPlayerName(game.getPlayerColor(WHITE)));
				
			} else {
				set = true;
			}
		}
		return set;
	}
	
	/**
	 * Displays the win dialog for whoever won.
	 * 
	 * @param winner the person who won
	 */
	private void showWin(String winner) {
		String msg = winner + " has won!";
		JOptionPane.showMessageDialog(this, msg, "WIN", JOptionPane.INFORMATION_MESSAGE);
		
		setDefaultCloseOperation(EXIT_ON_CLOSE);
		drawButton.setEnabled(false);
		resignButton.setEnabled(false);
	}
	
	/**
	 * Places all the pieces on the board, and Moves the pieces
	 *
	 */
	private void setPieces() {

		//go through all the squares
		for(int i = 0; i < NUM_SPACES; i++) {
			
			//if their is a piece on the square
			if(board.getPieceAt(i) != null) {
				
				//check if the piece is king or single
				checkType(i);

				//if there isn't a piece there, make sure there isn't an icon
			}else {
				((JButton)squares.get(i)).setIcon(null);
			}
			
		}
		
	}
	
	/**
	 * Check if the piece at a specific location is a King or a Single piece
	 * 
	 * @param location the square to be checked
	 */
	private void checkType(int location) {
		if(board.getPieceAt(location).getType() == SINGLE){
			checkColor(location, SINGLE);
		} else {
			checkColor(location, KING);
		}
	}
	
	/**
	 * Check what color a piece is and Display the correct one
	 * 
	 * @param location the square of the piece
	 * @param type the type of piece, King or Single
	 */
	private void checkColor(int location, int type) {
		//Create Icon for buttons to display images
		ImageIcon white = new ImageIcon("WhiteSingle.gif");
		ImageIcon blue  = new ImageIcon("BlueSingle.gif");
		ImageIcon kWhite = new ImageIcon("WhiteKing.gif");
		ImageIcon kBlue = new ImageIcon("BlueKing.gif");
		if(board.getPieceAt(location).getColor() == WHITE) {
			if(type == SINGLE) {
				((JButton)squares.get(location)).setIcon(white);
			}else{
				((JButton)squares.get(location)).setIcon(kWhite);
			}
		}else {
			if(type == SINGLE) {
				((JButton)squares.get(location)).setIcon(blue);
			}else{
				((JButton)squares.get(location)).setIcon(kBlue);
			}
		}
	}
	
	/**
	 * Used when a player resigns, anounces the message to all players
	 * 
	 * @param msg message to be anounced
	 * @param turn the winner
	 */
	
	private void resignGame(String msg, String turn){
		JOptionPane.showMessageDialog(this, msg, "End Game", JOptionPane.INFORMATION_MESSAGE);
		showWin(turn);
	}
	
	/**
	 * Makes the current players pieces send action events, and the other 
	 * player's buttons do nothing.
	 * 
	 * @param turn player num
	 */
	private void setTurn(boolean turn){
		for (int i = 0; i < NUM_SPACES; i++){
		
				if (turn){
					if (!actionListenerAdded){						
						((JButton)squares.get(i)).addActionListener(listener);
						resignButton.addActionListener(listener);
						drawButton.addActionListener(listener);
					}
				}else{
					if (actionListenerAdded){
						((JButton)squares.get(i)).removeActionListener(listener);
						resignButton.removeActionListener(listener);
						drawButton.removeActionListener(listener);
					}
				}
		
		}
		actionListenerAdded = turn;
	}
	
	/**
	 * Display a message to offer a draw
	 * 
	 * @return whether the draw was accepted or not
	 */
	public boolean offerDraw() {
		//the value of the response from the dialog box
		int value = JOptionPane.showOptionDialog(this, "Do you Accept a Draw?", 
				"Offer Draw", JOptionPane.DEFAULT_OPTION, JOptionPane.PLAIN_MESSAGE, 
				null, new Object[]{"Reject", "Accept"}, "Reject");
		
		return value == 1;
	}
	
	/**
	 * Display a message saying the game was a draw, end the game
	 *
	 */
	private void acceptDraw() {
		JOptionPane.showMessageDialog(this, "It's a Draw!", "End Game", 
				JOptionPane.INFORMATION_MESSAGE);
		setDefaultCloseOperation(EXIT_ON_CLOSE);
		drawButton.setEnabled(false);
		resignButton.setEnabled(false);
	}
	
	
}// checkerGUI.java
