/*
 * NetworkView.java
 *
 * Version:
 *     $Id: NetworkView.java,v 1.14 2005/11/08 23:02:10 pjk7060 Exp $
 *
 * Revisions:
 *     $Log: NetworkView.java,v $
 *     Revision 1.14  2005/11/08 23:02:10  pjk7060
 *     comments
 *
 *     Revision 1.13  2005/11/08 22:37:00  pjk7060
 *     removed unused timer code
 *
 *     fixed warnings
 *
 *     Revision 1.12  2005/11/08 05:13:05  mtg9625
 *     Removed Main
 *
 *     Revision 1.11  2005/11/07 02:43:27  pjk7060
 *     catch nasty exceptions
 *
 *     Revision 1.10  2005/11/06 23:38:31  pjk7060
 *     uses the CheckersConstants interface for constants
 *
 *     Revision 1.9  2005/11/06 22:46:08  pjk7060
 *     fixed resigning over the network
 *
 *     Revision 1.8  2005/11/06 22:12:51  pjk7060
 *     forgot a flush
 *
 *     Revision 1.7  2005/11/06 21:52:15  pjk7060
 *     fixed network play colors not being assigned correctly
 *
 *     Revision 1.6  2005/11/06 20:18:54  pjk7060
 *     Fixed turn issues
 *
 *     Revision 1.5  2005/11/06 19:38:46  pjk7060
 *     worked many bugs out of networking
 *
 *     Revision 1.4  2005/11/05 15:35:32  pjk7060
 *     Little fixes
 *
 *     Revision 1.3  2005/11/04 15:59:42  pjk7060
 *     *** empty log message ***
 *
 *     Revision 1.2  2005/11/02 04:34:05  pjk7060
 *     moved classes into the proper packages
 *
 *     Revision 1.1  2005/11/02 04:31:21  pjk7060
 *     split everything into packages
 *
 *     implemented the networking class
 *
 */
package checkers.view;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Observable;
import java.util.Observer;

import javax.swing.JOptionPane;

import checkers.CheckersConstants;
import checkers.controller.GameController;
import checkers.model.CheckersGame;
import checkers.model.Move;
import checkers.model.NetPacket;


/**
 * Observes model, and sends any changes over the network.  
 * 
 * When a change is received, it imitates the GUI by generating the same 
 * ActionEvents as if the change happened locally.
 * 
 * This makes networking transparent to the controller.
 * 
 * @author Peter Kuhn
 */
public class NetworkView extends Thread implements Observer, CheckersConstants {
	
	// Port to listen / connect via
	public static final int PORT_NUM = 1051; 
	
	// Listeners to update on performAction
	private List actionListeners = new ArrayList();
	
	// Socket to accept connections
	private ServerSocket socket;
	
	// The remote client to communicate with
	private Socket client = null;
	
	// whether or not we are currently connected
	private boolean isConnected = false;
	
	// stream to send network communication over
	ObjectOutputStream oos;
	
	// stream to receive network communicatio over
	ObjectInputStream ois;
	
	// Observables to register with
	private List observables = new ArrayList();
	
	// Name of the current player
	private String playerName;
	
	// Num of current player, to distinguish each side of network
	private int playerNum;
	
	// Hosting or Joining
	private boolean hostGame = false;
	
	// Helps make sure that the players are assigned to the right colors
	private int player1Color;
	
	/**
	 * Registers Actionlisteners and sets player1color
	 */
	public NetworkView(Observable a1, Observable a2, int color) {
		observables.add(a1);
		observables.add(a2);
		player1Color = color;
	}
	
	/**
	 * Connects to specified IP address and starts the network thread.
	 * 
	 * @param ip ip to connect to
	 */
	public void connectToHost(String ip){
		hostGame = false;
		setNetwork();
		try{
			client = new Socket(ip, PORT_NUM);			
			this.start();
		}catch(IOException e){
			errorHandler(e);
		}
	}
	
	/**
	 * Starts hosting a game and waits for a remote client to connect.
	 */
	public void hostGame() {
		hostGame = true;
		setNetwork();
		this.start();		
	}
	
	/**
	 * Cancel Host Listening without closing the program.
	 */
	public void stopHostListening(){
		try{
			socket.close();
		}catch(IOException e){
			// closing anyway, doesn't matter
		}
	}
	
	/**
	 * Registers networkView with observables.
	 */
	public void setNetwork(){
		Iterator it = observables.iterator();
		
		while (it.hasNext()){
			((Observable)it.next()).addObserver(this);
		}
	}
	
	/**
	 * Network thread, sets up the network and receives packets and handles 
	 * them appropriately.
	 */
	public void run(){
		try{
			// we're hosting, we need to wait for a connection
			if (client == null){
				log("Waiting for connections");
				socket = new ServerSocket(PORT_NUM);			
				client = socket.accept();
				log("Accepted a connection from: " + client.getInetAddress());
			}
			
			// connection is initialized at this point			
			isConnected = true;
			
			// initialized the in and out streams
			oos = new ObjectOutputStream( client.getOutputStream() );
			ois = new ObjectInputStream( client.getInputStream() );
			
			oos.writeObject(getNamePacket());
			oos.flush();
			
			// accept a NetPacket from the remote player
			NetPacket input = (NetPacket)ois.readObject();
			
			while (isConnected && client.isConnected()){
				
				
				
				log("Packet received: " + input.toString());
				
				// handle it appropriately
				if (input.isDraw()){
					offerDraw();
				}else if (input.isResign()){
					resign();
				}else if (input.getMove() != null){
					makeMove(input.getMove());
				}else if (input.getPlayer1() != null){
					ActionEvent ae = new ActionEvent(input, 0, PLAYER1SET);
					performAction(ae);
				}else if (input.getPlayer2() != null){
					ActionEvent ae = new ActionEvent(input, 0, PLAYER2SET);
					performAction(ae);
				}
				
				if (isConnected){
					try{
						input = (NetPacket)ois.readObject();
					}catch(SocketException e){
						// it's ok
					}
				}else{
					break;
				}
				
			}
			
			log("Disconnected");
			
			// close the streams, we're done
			ois.close();
			oos.close();
			client.close();			
			
		}catch(IOException e){
			e.printStackTrace();
		}catch(ClassNotFoundException e){
			e.printStackTrace();
		}
	}
	
	/**
	 * Allows for spoofed Actions to be sent to the controller
	 * 
	 * @param a controller
	 */
	public void addActionListener(ActionListener a){
		actionListeners.add(a);
	}
	
	/**
	 * Recievees updates from the model and sends them over the network.
	 * 
	 * @param arg0
	 * @param arg1
	 */
	public void update(Observable arg0, Object arg1) {
		
		if (isConnected() && arg1 != null){
			NetPacket np = new NetPacket();			
			
			// determine what should be in the NetPacket based on the update
			if (arg1 instanceof Move){
				np.setMove((Move)arg1);
			}else if (arg1 instanceof String){
				if (arg1.equals(RESIGN)){
					np.setResign(true);
				}else if (arg1.equals(DRAW)){
					np.setDraw(true);
				}
			}
			
			// send the object over the network
			try{
				oos.writeObject(np);
				oos.flush();
				log("Sent Packet: " + np);
			}catch(IOException e){
				errorHandler(e);
			}	
			
			if (arg0 instanceof CheckersGame){
				CheckersGame game = (CheckersGame)arg0;
				
				setConnected(game.isGameActive());
			}
		}
	}
	
	/**
	 * Acts just like a GUI object performing an action.
	 * 
	 * @param e spoofed action event
	 */
	private void performAction(ActionEvent e){
		Iterator it = actionListeners.iterator();
		
		while (it.hasNext()){
			((ActionListener)it.next()).actionPerformed(e);
		}
	}
	
	/**
	 * Imitates clicking the draw button.
	 */
	private void offerDraw(){
		ActionEvent ae = new ActionEvent(this, 0, GameController.DRAW);
		performAction(ae);
	}
	
	/**
	 * Imitates clicking the resign button.
	 */
	private void resign(){
		ActionEvent ae = new ActionEvent(this, 0, GameController.RESIGN);
		performAction(ae);		
	}
	
	/**
	 * Imitates making a move, by splitting it into two separate events 
	 * (the first click and the second click)
	 * 
	 * @param m Move to imitate
	 */
	private void makeMove(Move m){
		// Immitate clicking the first square
		ActionEvent ae = new ActionEvent(this, 0 , String.valueOf(m.startLocation()));
		performAction(ae);
		
		// Imitate clicking the second square
		ae = new ActionEvent(this, 0, String.valueOf(m.endLocation()));
		performAction(ae);
	}
	
	/**
	 * Outputs any logging related messages.
	 * Format: [time] message
	 * 
	 * @param s Message
	 */
	private void log(String s){
		System.out.println(getTime() + s);
	}
	
	/**
	 * Returns the current time in the format [HH:mm:ss]
	 * 
	 * @return Current time
	 */
	private String getTime(){
		SimpleDateFormat s = new SimpleDateFormat("HH:mm:ss");
		return "[" + s.format(new Date(System.currentTimeMillis())) + "] ";
	}
	
	/**
	 * Private method to handle any connection errors.
	 * 
	 * @param e The Exception that was thrown (not used at the moment)
	 */
	private static void errorHandler(Exception e){
		JOptionPane.showMessageDialog(null, 
						"Error connecting to server.\n\n" + e.getMessage(), "Error", JOptionPane.ERROR_MESSAGE);
	}
	
	/**
	 * @return Returns the isConnected.
	 */
	private boolean isConnected() {
		return isConnected;
	}

	/**
	 * @param isConnected The isConnected to set.
	 */
	private void setConnected(boolean isConnected) {
		this.isConnected = isConnected;
	}

	/**
	 * Sets the player's name so it can be sent over the network on initial 
	 * handshake.
	 * 
	 * @param num player num
	 * @param name player name
	 */
	public void setPlayerName(int num, String name){
		this.playerName = name;
		this.playerNum = num;
	}
	
	/**
	 * Returns the packet to send
	 * 
	 * @return
	 */
	private NetPacket getNamePacket(){
		NetPacket np = new NetPacket();
		if (playerNum == 0){
			np.setPlayer1(playerName);
		}else if (playerNum == 1){
			np.setPlayer2(playerName);
		}
		
		if (hostGame){
			np.setPlayer1Color(player1Color);
		}
		
		return np;
	}
}
